<?php
declare(strict_types=1);

ini_set('display_errors', '1');
ini_set('display_startup_errors', '1');
error_reporting(E_ALL);

// blog_add.php

if (session_status() !== PHP_SESSION_ACTIVE)
    session_start();

require_once __DIR__ . '/includes/db.php';
require_once __DIR__ . '/includes/auth.php';

// Require authentication first
if (empty($_SESSION['user_id'])) {
    header('Location: /login.php');
    exit;
}

// Check if user is admin OR expert (adjust roles as needed for blog authoring)
$stmt = $conn->prepare("SELECT role, forum_designation FROM users WHERE id = ? LIMIT 1");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$stmt->close();

if (!$user || ($user['role'] !== 'admin' && $user['forum_designation'] !== 'Expert')) {
    http_response_code(403);
    exit('Access denied. Only admins and experts can add blog posts.');
}

// CSRF
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

function csrf_verify(): void
{
    $posted = $_POST['csrf_token'] ?? '';
    if (!is_string($posted) || !hash_equals($_SESSION['csrf_token'] ?? '', $posted)) {
        http_response_code(400);
        exit('Bad request (CSRF).');
    }
}

// Handle image upload directory
$uploadDir = __DIR__ . '/uploads/blog_images/';
if (!is_dir($uploadDir)) {
    mkdir($uploadDir, 0755, true);
}

// Handle POST submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    csrf_verify();

    $title = trim($_POST['title'] ?? '');
    $slug = trim($_POST['slug'] ?? '');
    $content = $_POST['content'] ?? '';
    $is_published = isset($_POST['is_published']) ? 1 : 0;
    $default_color = trim($_POST['default_color'] ?? '#adb5bd'); // Optional accent color
    $imagePath = null;

    // Handle image upload
    if (isset($_FILES['image']) && $_FILES['image']['error'] === UPLOAD_ERR_OK) {
        $fileTmpPath = $_FILES['image']['tmp_name'];
        $fileName = $_FILES['image']['name'];
        $fileSize = $_FILES['image']['size'];
        $fileType = $_FILES['image']['type'];
        $fileNameCmps = explode(".", $fileName);
        $fileExtension = strtolower(end($fileNameCmps));

        // Validate file type and size
        $allowedExts = ['jpg', 'jpeg', 'png', 'gif'];
        if (in_array($fileExtension, $allowedExts) && $fileSize <= 2097152) { // 2MB max
            $newFileName = time() . '_' . bin2hex(random_bytes(4)) . '.' . $fileExtension;
            $destPath = $uploadDir . $newFileName;
            if (move_uploaded_file($fileTmpPath, $destPath)) {
                $imagePath = 'uploads/blog_images/' . $newFileName;
            } else {
                $error = "Failed to upload image.";
            }
        } else {
            $error = "Invalid image file. Only JPG, PNG, GIF under 2MB allowed.";
        }
    } elseif (isset($_FILES['image']) && $_FILES['image']['error'] !== UPLOAD_ERR_NO_FILE) {
        $error = "Image upload error: " . $_FILES['image']['error'];
    }

    // Validate default color (hex only)
    if (!preg_match('/^#[a-fA-F0-9]{6}$/', $default_color)) {
        $default_color = '#adb5bd'; // Fallback
    }

    // Auto-generate slug if empty
    if ($slug === '' && $title !== '') {
        $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title), '-'));
    }

    // Validate
    if ($title === '' || trim(strip_tags($content)) === '') {
        $error = "Title and content are required.";
    } elseif ($slug === '') {
        $error = "Slug could not be generated.";
    } elseif (isset($error)) {
        // Error already set above
    } else {
        // Check if slug already exists
        $stmt = $conn->prepare("SELECT id FROM blog_posts WHERE slug = ? LIMIT 1");
        $stmt->bind_param("s", $slug);
        $stmt->execute();
        $result = $stmt->get_result();
        if ($result->num_rows > 0) {
            $error = "A post with this slug already exists. Please use a different slug.";
            $stmt->close();
        } else {
            $stmt->close();

            // Insert new post
            $stmt = $conn->prepare("
                INSERT INTO blog_posts 
                (title, slug, content, image, is_published, default_color, created_at, updated_at)
                VALUES (?, ?, ?, ?, ?, ?, NOW(), NOW())
            ");
            if (!$stmt) {
                $error = "Prepare failed: " . $conn->error;
            } else {
                $stmt->bind_param("ssssiss", $title, $slug, $content, $imagePath, $is_published, $default_color);
                if ($stmt->execute()) {
                    $stmt->close();
                    header("Location: blog.php");
                    exit;
                }
                $error = "Save failed: " . $stmt->error;
                $stmt->close();
            }
        }
    }
}

$pageTitle = "Add Blog Post";

require_once __DIR__ . '/includes/header.php';
?>

<div class="container py-4">
  <div class="d-flex align-items-center justify-content-between mb-3">
    <h1 class="h4 mb-0">Add New Blog Post</h1>
    <a class="btn btn-sm btn-outline-secondary" href="blog.php">Back</a>
  </div>

  <?php if (!empty($error)): ?>
    <div class="alert alert-danger"><?= h($error) ?></div>
  <?php endif; ?>

  <form method="post" enctype="multipart/form-data">
    <input type="hidden" name="csrf_token" value="<?= h($csrf) ?>">

    <div class="mb-3">
      <label class="form-label">Featured Image (JPG, PNG, GIF - max 2MB)</label>
      <input type="file" name="image" class="form-control" accept=".jpg,.jpeg,.png,.gif">
      <div class="form-text">Optional. Will be displayed at the top of the post preview.</div>
    </div>

    <div class="mb-3">
      <label class="form-label">Title</label>
      <input type="text" name="title" class="form-control" required value="<?= h($_POST['title'] ?? '') ?>">
    </div>

    <div class="mb-3">
      <label class="form-label">Slug (optional)</label>
      <input type="text" name="slug" class="form-control" value="<?= h($_POST['slug'] ?? '') ?>">
      <div class="form-text">Leave empty to auto-generate from title. Use lowercase + hyphens.</div>
    </div>

    <div class="mb-3">
      <label class="form-label">Accent Color</label>
      <input type="text" name="default_color" class="form-control" value="<?= h($_POST['default_color'] ?? '#adb5bd') ?>" placeholder="#adb5bd">
      <div class="form-text">Hex color for the post card border (optional, defaults to gray).</div>
    </div>

    <div class="mb-3">
      <label class="form-label">Content</label>
      <textarea name="content" id="editor" rows="10" data-ckeditor="1" class="form-control"><?= h($_POST['content'] ?? '') ?></textarea>
    </div>

    <div class="form-check mb-3">
      <input class="form-check-input" type="checkbox" name="is_published" id="is_published"
        <?= (isset($_POST['is_published'])) ? 'checked' : '' ?>>
      <label class="form-check-label" for="is_published">Published</label>
    </div>

    <button class="btn btn-primary">Create Post</button>
  </form>
</div>

<!-- CKEditor: update path to your actual installed file -->
<script src="/path/to/ckeditor.js"></script>
<script>
  ClassicEditor.create(document.querySelector('#editor'))
    .catch(error => console.error(error));
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>