# update-schema.ps1
# FULLY AUTOMATIC - Detects all tables and dependencies
# Add new tables to your database - script handles everything

$ErrorActionPreference = "Stop"

# Config
$mysqldump = "D:\ledger_server\bin\mariadb\bin\mysqldump.exe"
$mysql = "D:\ledger_server\bin\mariadb\bin\mysql.exe"
$schemaFile = "D:\ledger_server\site\install\schema.sql"
$dbHost = "127.0.0.1"
$dbPort = "3307"
$dbUser = "ledger_user"
$dbPass = "ledger123"
$dbName = "ledger_db"

Write-Host "=== AUTOMATIC Schema Dumper ===" -ForegroundColor Cyan
Write-Host ""

# Get all tables
Write-Host "Step 1: Finding all tables..." -ForegroundColor Green
$getTables = "SELECT TABLE_NAME FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA='$dbName' AND TABLE_TYPE='BASE TABLE' ORDER BY TABLE_NAME"
$rawTables = & $mysql --host=$dbHost --port=$dbPort --user=$dbUser --password=$dbPass --batch --skip-column-names -e $getTables 2>&1

# Filter out warnings
$allTables = @()
foreach ($line in $rawTables) {
    if ($line -match '^\s*$') { continue }
    if ($line -match 'mysql:.*Warning') { continue }
    $allTables += $line.Trim()
}

Write-Host "Found $($allTables.Count) tables:" -ForegroundColor Cyan
$allTables | ForEach-Object { Write-Host "  $_" -ForegroundColor Gray }
Write-Host ""

# Get foreign keys
Write-Host "Step 2: Detecting dependencies..." -ForegroundColor Green
$getFK = "SELECT TABLE_NAME, REFERENCED_TABLE_NAME FROM INFORMATION_SCHEMA.KEY_COLUMN_USAGE WHERE TABLE_SCHEMA='$dbName' AND REFERENCED_TABLE_NAME IS NOT NULL"
$rawFK = & $mysql --host=$dbHost --port=$dbPort --user=$dbUser --password=$dbPass --batch --skip-column-names -e $getFK 2>&1

# Parse foreign keys
$deps = @{}
foreach ($line in $rawFK) {
    if ($line -match '^\s*$') { continue }
    if ($line -match 'mysql:.*Warning') { continue }
    
    $parts = $line.Trim() -split '\t'
    if ($parts.Length -eq 2) {
        $child = $parts[0]
        $parent = $parts[1]
        
        # Skip self-referencing
        if ($child -eq $parent) {
            Write-Host "  $child (self-referencing, will go last)" -ForegroundColor Yellow
            continue
        }
        
        if (-not $deps[$child]) {
            $deps[$child] = @()
        }
        $deps[$child] += $parent
    }
}

if ($deps.Count -gt 0) {
    Write-Host "Dependencies:" -ForegroundColor Cyan
    foreach ($t in $deps.Keys | Sort-Object) {
        Write-Host "  $t -> $($deps[$t] -join ', ')" -ForegroundColor Gray
    }
} else {
    Write-Host "  (no dependencies detected)" -ForegroundColor Gray
}
Write-Host ""

# Sort tables
Write-Host "Step 3: Sorting tables..." -ForegroundColor Green
$ordered = @()
$visited = @{}

function Add-Table($name) {
    if ($visited[$name]) { return }
    
    # Add dependencies first
    if ($deps[$name]) {
        foreach ($parent in $deps[$name]) {
            Add-Table $parent
        }
    }
    
    $visited[$name] = $true
    $script:ordered += $name
}

foreach ($table in $allTables) {
    Add-Table $table
}

Write-Host "Order:" -ForegroundColor Cyan
$num = 1
$ordered | ForEach-Object { 
    Write-Host "  $num. $_" -ForegroundColor Gray
    $num++
}
Write-Host ""

# Backup
if (Test-Path $schemaFile) {
    Copy-Item $schemaFile "$schemaFile.backup" -Force
    Write-Host "Backed up existing file" -ForegroundColor Yellow
}

# Dump
Write-Host "Step 4: Dumping schema..." -ForegroundColor Green
$dumpOutput = & $mysqldump --host=$dbHost --port=$dbPort --user=$dbUser --password=$dbPass --no-data --skip-comments --skip-add-drop-table --routines --triggers $dbName $ordered 2>&1

# Write output (filter warnings)
$cleanOutput = $dumpOutput | Where-Object { $_ -notmatch 'mysql:.*Warning' }
$cleanOutput | Out-File -FilePath $schemaFile -Encoding UTF8

Write-Host ""
if ($LASTEXITCODE -eq 0) {
    $kb = [math]::Round((Get-Item $schemaFile).Length / 1KB, 2)
    Write-Host "SUCCESS! Schema dumped" -ForegroundColor Green
    Write-Host "File: $schemaFile" -ForegroundColor Cyan
    Write-Host "Size: $kb KB" -ForegroundColor Cyan
    Write-Host ""
    Write-Host "Tables will install in the correct order!" -ForegroundColor Green
} else {
    Write-Host "ERROR: Dump failed" -ForegroundColor Red
}

Write-Host ""
Read-Host "Press Enter to close"