Set-ExecutionPolicy -Scope Process -ExecutionPolicy Bypass -Force

# ============================================================================
# Ledger Server Startup Script (DEBUG VERSION)
# Starts MariaDB, PHP-FPM, and Caddy web server
# ============================================================================

$root = Split-Path -Parent $PSScriptRoot
$logsDir = Join-Path $root "logs"
New-Item -ItemType Directory -Force -Path $logsDir | Out-Null
$runLog = Join-Path $logsDir "start_stack.log"

$ErrorActionPreference = "Stop"

function Log($msg, $color = "White") {
  $line = "[{0}] {1}" -f (Get-Date -Format "HH:mm:ss"), $msg
  Write-Host $line -ForegroundColor $color
  Add-Content -Path $runLog -Value $line
}

function Test-PathAndLog($path, $description) {
  if (Test-Path $path) {
    Log "? Found: $description" "Green"
    Log "  Path: $path" "DarkGray"
    return $true
  } else {
    Log "? MISSING: $description" "Red"
    Log "  Expected path: $path" "DarkGray"
    return $false
  }
}

function Test-Port($port, $serviceName) {
  try {
    $connection = New-Object System.Net.Sockets.TcpClient
    $connection.Connect("127.0.0.1", $port)
    $connection.Close()
    Log "? Port $port is already in use (might conflict with $serviceName)" "Yellow"
    return $true
  } catch {
    Log "? Port $port is available for $serviceName" "Green"
    return $false
  }
}

function Wait-ForPort($port, $serviceName, $timeoutSeconds = 10) {
  Log "Waiting for $serviceName to listen on port $port..." "Cyan"
  $elapsed = 0
  while ($elapsed -lt $timeoutSeconds) {
    try {
      $connection = New-Object System.Net.Sockets.TcpClient
      $connection.Connect("127.0.0.1", $port)
      $connection.Close()
      Log "? $serviceName is responding on port $port" "Green"
      return $true
    } catch {
      Start-Sleep -Milliseconds 500
      $elapsed += 0.5
    }
  }
  Log "? $serviceName did not respond on port $port within $timeoutSeconds seconds" "Red"
  return $false
}

try {
  Log "=== STARTING LEDGER SERVER (DEBUG MODE) ===" "Cyan"
  Log "PowerShell Version: $($PSVersionTable.PSVersion)" "Gray"
  Log "Root directory: $root" "Gray"
  Log "Log file: $runLog" "Gray"
  Write-Host ""

  # ===== ENVIRONMENT CHECK =====
  Log "=== CHECKING ENVIRONMENT ===" "Cyan"
  Log "Current user: $env:USERNAME" "Gray"
  Log "Computer name: $env:COMPUTERNAME" "Gray"
  Log "Current directory: $PWD" "Gray"
  Write-Host ""

  # ===== PATH VALIDATION =====
  Log "=== VALIDATING ALL PATHS ===" "Cyan"
  
  # Define all paths
  $caddyExe   = Join-Path $root "bin\caddy.exe"
  $caddyFile  = Join-Path $root "conf\Caddyfile"
  $phpDir     = Join-Path $root "php"
  $phpExe     = Join-Path $phpDir "php-cgi.exe"
  $phpIni     = Join-Path $root "conf\php.ini"
  $mariaRoot  = Join-Path $root "bin\mariadb"
  $mariaBin   = Join-Path $mariaRoot "bin"
  $mysqld     = Join-Path $mariaBin "mysqld.exe"
  $mariadbd   = Join-Path $mariaBin "mariadbd.exe"
  $dataDir    = Join-Path $mariaRoot "data"
  $dataMyIni  = Join-Path $dataDir "my.ini"

  # Check all paths
  $pathsValid = $true
  $pathsValid = (Test-PathAndLog $root "Root directory") -and $pathsValid
  $pathsValid = (Test-PathAndLog $logsDir "Logs directory") -and $pathsValid
  $pathsValid = (Test-PathAndLog $caddyExe "Caddy executable") -and $pathsValid
  $pathsValid = (Test-PathAndLog $caddyFile "Caddyfile") -and $pathsValid
  $pathsValid = (Test-PathAndLog $phpDir "PHP directory") -and $pathsValid
  $pathsValid = (Test-PathAndLog $phpExe "PHP-CGI executable") -and $pathsValid
  $pathsValid = (Test-PathAndLog $phpIni "PHP configuration") -and $pathsValid
  $pathsValid = (Test-PathAndLog $mariaRoot "MariaDB root") -and $pathsValid
  $pathsValid = (Test-PathAndLog $mariaBin "MariaDB bin directory") -and $pathsValid
  $pathsValid = (Test-PathAndLog $dataDir "MariaDB data directory") -and $pathsValid
  $pathsValid = (Test-PathAndLog $dataMyIni "MariaDB my.ini") -and $pathsValid

  if (-not $pathsValid) {
    throw "One or more required paths are missing. Check the logs above."
  }
  
  Write-Host ""

  # ===== PORT CHECK =====
  Log "=== CHECKING PORTS ===" "Cyan"
  Test-Port 3307 "MariaDB"
  Test-Port 9000 "PHP-FPM"
  Test-Port 8443 "Caddy HTTPS"
  Test-Port 8080 "Caddy HTTP"
  Write-Host ""

  # ===== PROCESS CHECK =====
  Log "=== CHECKING FOR EXISTING PROCESSES ===" "Cyan"
  $existingProcesses = @()
  
  $mysqldProcess = Get-Process -Name "mysqld" -ErrorAction SilentlyContinue
  if ($mysqldProcess) {
    Log "? Found existing mysqld process (PID: $($mysqldProcess.Id))" "Yellow"
    $existingProcesses += "mysqld"
  }
  
  $mariadbdProcess = Get-Process -Name "mariadbd" -ErrorAction SilentlyContinue
  if ($mariadbdProcess) {
    Log "? Found existing mariadbd process (PID: $($mariadbdProcess.Id))" "Yellow"
    $existingProcesses += "mariadbd"
  }
  
  $phpProcess = Get-Process -Name "php-cgi" -ErrorAction SilentlyContinue
  if ($phpProcess) {
    Log "? Found existing php-cgi process (PID: $($phpProcess.Id))" "Yellow"
    $existingProcesses += "php-cgi"
  }
  
  $caddyProcess = Get-Process -Name "caddy" -ErrorAction SilentlyContinue
  if ($caddyProcess) {
    Log "? Found existing caddy process (PID: $($caddyProcess.Id))" "Yellow"
    $existingProcesses += "caddy"
  }

  if ($existingProcesses.Count -eq 0) {
    Log "? No conflicting processes found" "Green"
  } else {
    Log "? Warning: Found $($existingProcesses.Count) existing process(es). This may cause issues." "Yellow"
  }
  Write-Host ""

  # ===== START MARIADB =====
  Log "=== STARTING MARIADB ===" "Cyan"
  
  # Determine which MariaDB binary exists
  if (Test-Path $mysqld) { 
    $server = $mysqld 
    Log "Using mysqld.exe" "Gray"
  }
  elseif (Test-Path $mariadbd) { 
    $server = $mariadbd 
    Log "Using mariadbd.exe" "Gray"
  }
  else { 
    throw "MariaDB binary not found in $mariaBin. Checked for mysqld.exe and mariadbd.exe"
  }

  Log "Server binary: $server" "Gray"
  Log "Config file: $dataMyIni" "Gray"
  Log "Working directory: $mariaBin" "Gray"
  
  # Display my.ini contents for debugging
  if (Test-Path $dataMyIni) {
    Log "Reading my.ini configuration..." "Gray"
    $myIniContent = Get-Content $dataMyIni -Raw
    Log "my.ini contents:" "Gray"
    $myIniContent -split "`n" | ForEach-Object { Log "  $_" "DarkGray" }
  }

  try {
    Log "Executing: Start-Process -FilePath $server -ArgumentList --defaults-file=`"$dataMyIni`"" "Gray"
    $mariaProcess = Start-Process -FilePath $server `
      -ArgumentList "--defaults-file=`"$dataMyIni`"" `
      -WorkingDirectory $mariaBin `
      -WindowStyle Hidden `
      -PassThru
    
    Log "MariaDB process started (PID: $($mariaProcess.Id))" "Green"
    
    # Wait and verify
    Start-Sleep -Seconds 2
    
    # Check if process is still running
    $processCheck = Get-Process -Id $mariaProcess.Id -ErrorAction SilentlyContinue
    if ($processCheck) {
      Log "? MariaDB process is running" "Green"
    } else {
      throw "MariaDB process died immediately after starting. Check MariaDB error logs in $dataDir"
    }
    
    # Try to connect to port 3307
    Wait-ForPort 3307 "MariaDB" 10
    
  } catch {
    Log "? Failed to start MariaDB: $($_.Exception.Message)" "Red"
    throw "MariaDB startup failed: $($_.Exception.Message)"
  }
  
  Write-Host ""

  # ===== START PHP-FPM =====
  Log "=== STARTING PHP FASTCGI ===" "Cyan"
  Log "PHP executable: $phpExe" "Gray"
  Log "PHP config: $phpIni" "Gray"
  Log "Working directory: $phpDir" "Gray"
  
  try {
    Log "Executing: Start-Process -FilePath $phpExe -ArgumentList -b 127.0.0.1:9000 -c `"$phpIni`"" "Gray"
    $phpProcess = Start-Process -FilePath $phpExe `
      -ArgumentList "-b 127.0.0.1:9000 -c `"$phpIni`"" `
      -WorkingDirectory $phpDir `
      -WindowStyle Hidden `
      -PassThru
    
    Log "PHP-CGI process started (PID: $($phpProcess.Id))" "Green"
    
    # Wait and verify
    Start-Sleep -Seconds 1
    
    $processCheck = Get-Process -Id $phpProcess.Id -ErrorAction SilentlyContinue
    if ($processCheck) {
      Log "? PHP-CGI process is running" "Green"
    } else {
      throw "PHP-CGI process died immediately after starting"
    }
    
    # Try to connect to port 9000
    Wait-ForPort 9000 "PHP-CGI" 5
    
  } catch {
    Log "? Failed to start PHP-CGI: $($_.Exception.Message)" "Red"
    throw "PHP-CGI startup failed: $($_.Exception.Message)"
  }
  
  Write-Host ""

  # ===== START CADDY =====
  Log "=== STARTING CADDY WEB SERVER ===" "Cyan"
  Log "Caddy executable: $caddyExe" "Gray"
  Log "Caddyfile: $caddyFile" "Gray"
  Log "Working directory: $root" "Gray"
  
  # Set environment variable for Caddyfile
  $env:LEDGER_ROOT = $root
  Log "Environment: LEDGER_ROOT=$env:LEDGER_ROOT" "Gray"
  
  # Display Caddyfile contents for debugging
  if (Test-Path $caddyFile) {
    Log "Reading Caddyfile configuration..." "Gray"
    $caddyContent = Get-Content $caddyFile -Raw
    Log "Caddyfile contents:" "Gray"
    $caddyContent -split "`n" | ForEach-Object { Log "  $_" "DarkGray" }
  }
  
  try {
    Log "Executing: Start-Process -FilePath $caddyExe -ArgumentList run --config `"$caddyFile`" --adapter caddyfile" "Gray"
    $caddyProcess = Start-Process -FilePath $caddyExe `
      -ArgumentList "run --config `"$caddyFile`" --adapter caddyfile" `
      -WorkingDirectory $root `
      -WindowStyle Hidden `
      -PassThru
    
    Log "Caddy process started (PID: $($caddyProcess.Id))" "Green"
    
    # Wait and verify
    Start-Sleep -Seconds 2
    
    $processCheck = Get-Process -Id $caddyProcess.Id -ErrorAction SilentlyContinue
    if ($processCheck) {
      Log "? Caddy process is running" "Green"
    } else {
      throw "Caddy process died immediately after starting. Check Caddyfile syntax."
    }
    
    # Try to connect to port 8443
    Wait-ForPort 8443 "Caddy HTTPS" 10
    
  } catch {
    Log "? Failed to start Caddy: $($_.Exception.Message)" "Red"
    throw "Caddy startup failed: $($_.Exception.Message)"
  }
  
  Write-Host ""

  # ===== ALL SERVICES STARTED =====
  Log "=== FINAL STATUS CHECK ===" "Cyan"
  
  $allRunning = $true
  
  # Check MariaDB
  $mariaCheck = Get-Process -Name "mysqld","mariadbd" -ErrorAction SilentlyContinue
  if ($mariaCheck) {
    Log "? MariaDB is running (PID: $($mariaCheck.Id))" "Green"
  } else {
    Log "? MariaDB is not running" "Red"
    $allRunning = $false
  }
  
  # Check PHP
  $phpCheck = Get-Process -Name "php-cgi" -ErrorAction SilentlyContinue
  if ($phpCheck) {
    Log "? PHP-CGI is running (PID: $($phpCheck.Id))" "Green"
  } else {
    Log "? PHP-CGI is not running" "Red"
    $allRunning = $false
  }
  
  # Check Caddy
  $caddyCheck = Get-Process -Name "caddy" -ErrorAction SilentlyContinue
  if ($caddyCheck) {
    Log "? Caddy is running (PID: $($caddyCheck.Id))" "Green"
  } else {
    Log "? Caddy is not running" "Red"
    $allRunning = $false
  }
  
  Write-Host ""
  
  if ($allRunning) {
    Write-Host "?????????????????????????????????????????" -ForegroundColor Green
    Write-Host "  SERVER READY" -ForegroundColor Green
    Write-Host "?????????????????????????????????????????" -ForegroundColor Green
    Write-Host ""
    Write-Host "  https://ledger.test:8443/" -ForegroundColor Yellow
    Write-Host ""
    
    Log "All services started successfully"

    # Open browser
    try {
      Start-Process "https://ledger.test:8443/"
      Log "Browser launched" "Green"
    } catch {
      Log "Could not launch browser: $($_.Exception.Message)" "Yellow"
    }
  } else {
    throw "One or more services failed to start properly"
  }

} catch {
  Write-Host ""
  Write-Host "?????????????????????????????????????????" -ForegroundColor Red
  Write-Host "  STARTUP FAILED" -ForegroundColor Red
  Write-Host "?????????????????????????????????????????" -ForegroundColor Red
  Write-Host ""
  Write-Host "Error: $($_.Exception.Message)" -ForegroundColor Red
  Write-Host ""
  Write-Host "Stack trace:" -ForegroundColor Yellow
  Write-Host $_.ScriptStackTrace -ForegroundColor Gray
  Write-Host ""
  Log "FAILED: $($_.Exception.Message)"
  Log "Stack trace: $($_.ScriptStackTrace)"
  
  Write-Host ""
  Write-Host "DEBUG TIPS:" -ForegroundColor Cyan
  Write-Host "1. Check the log file: $runLog" -ForegroundColor Gray
  Write-Host "2. Verify all file paths exist" -ForegroundColor Gray
  Write-Host "3. Check if ports 3307, 9000, 8443 are available" -ForegroundColor Gray
  Write-Host "4. Look for error logs in MariaDB data directory" -ForegroundColor Gray
  Write-Host "5. Try running services manually to see detailed errors" -ForegroundColor Gray
  
} finally {
  Write-Host ""
  Write-Host "Press any key to close..." -ForegroundColor Gray
  $null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
}