Set-ExecutionPolicy -Scope Process -ExecutionPolicy Bypass -Force

# ============================================================================
# Ledger Server Stop Script
# Stops MariaDB, PHP-CGI, and Caddy web server
# ============================================================================

$root = Split-Path -Parent $PSScriptRoot
$logsDir = Join-Path $root "logs"
New-Item -ItemType Directory -Force -Path $logsDir | Out-Null
$runLog = Join-Path $logsDir "stop_stack.log"

function Log($msg, $color = "White") {
  $line = "[{0}] {1}" -f (Get-Date -Format "HH:mm:ss"), $msg
  Write-Host $line -ForegroundColor $color
  Add-Content -Path $runLog -Value $line
}

Write-Host ""
Write-Host "=====================================" -ForegroundColor Cyan
Write-Host "  STOPPING LEDGER SERVER" -ForegroundColor Cyan
Write-Host "=====================================" -ForegroundColor Cyan
Write-Host ""

Log "=== STOPPING ALL SERVICES ===" "Cyan"

$totalStopped = 0
$totalFailed = 0

# ===== STOP CADDY =====
Write-Host "Stopping Caddy..." -ForegroundColor Yellow
$caddyProcesses = Get-Process -Name "caddy" -ErrorAction SilentlyContinue

if ($caddyProcesses) {
    foreach ($proc in $caddyProcesses) {
        try {
            Log "Stopping Caddy (PID: $($proc.Id))" "Gray"
            Stop-Process -Id $proc.Id -Force
            $totalStopped++
            Write-Host "  ? Stopped PID $($proc.Id)" -ForegroundColor Green
        } catch {
            Log "Failed to stop Caddy PID $($proc.Id): $($_.Exception.Message)" "Red"
            Write-Host "  ? Failed to stop PID $($proc.Id)" -ForegroundColor Red
            $totalFailed++
        }
    }
} else {
    Write-Host "  � Caddy not running" -ForegroundColor Gray
}

# ===== STOP PHP-CGI =====
Write-Host ""
Write-Host "Stopping PHP-CGI..." -ForegroundColor Yellow
$phpProcesses = Get-Process -Name "php-cgi","php" -ErrorAction SilentlyContinue

if ($phpProcesses) {
    foreach ($proc in $phpProcesses) {
        try {
            Log "Stopping PHP (PID: $($proc.Id))" "Gray"
            Stop-Process -Id $proc.Id -Force
            $totalStopped++
            Write-Host "  ? Stopped PID $($proc.Id)" -ForegroundColor Green
        } catch {
            Log "Failed to stop PHP PID $($proc.Id): $($_.Exception.Message)" "Red"
            Write-Host "  ? Failed to stop PID $($proc.Id)" -ForegroundColor Red
            $totalFailed++
        }
    }
} else {
    Write-Host "  � PHP-CGI not running" -ForegroundColor Gray
}

# ===== STOP MARIADB =====
Write-Host ""
Write-Host "Stopping MariaDB..." -ForegroundColor Yellow
$mariaProcesses = Get-Process -Name "mysqld","mariadbd" -ErrorAction SilentlyContinue

if ($mariaProcesses) {
    foreach ($proc in $mariaProcesses) {
        try {
            Log "Stopping MariaDB (PID: $($proc.Id))" "Gray"
            Stop-Process -Id $proc.Id -Force
            $totalStopped++
            Write-Host "  ? Stopped PID $($proc.Id)" -ForegroundColor Green
        } catch {
            Log "Failed to stop MariaDB PID $($proc.Id): $($_.Exception.Message)" "Red"
            Write-Host "  ? Failed to stop PID $($proc.Id)" -ForegroundColor Red
            $totalFailed++
        }
    }
} else {
    Write-Host "  � MariaDB not running" -ForegroundColor Gray
}

# Wait for processes to fully terminate
Start-Sleep -Seconds 2

# ===== VERIFY ALL STOPPED =====
Write-Host ""
Write-Host "Verifying services stopped..." -ForegroundColor Cyan

$stillRunning = @()

$caddyCheck = Get-Process -Name "caddy" -ErrorAction SilentlyContinue
if ($caddyCheck) {
    $stillRunning += "Caddy (PID: $($caddyCheck.Id))"
    Write-Host "  ? Caddy still running (PID: $($caddyCheck.Id))" -ForegroundColor Yellow
}

$phpCheck = Get-Process -Name "php-cgi","php" -ErrorAction SilentlyContinue
if ($phpCheck) {
    $stillRunning += "PHP-CGI (PID: $($phpCheck.Id))"
    Write-Host "  ? PHP-CGI still running (PID: $($phpCheck.Id))" -ForegroundColor Yellow
}

$mariaCheck = Get-Process -Name "mysqld","mariadbd" -ErrorAction SilentlyContinue
if ($mariaCheck) {
    $stillRunning += "MariaDB (PID: $($mariaCheck.Id))"
    Write-Host "  ? MariaDB still running (PID: $($mariaCheck.Id))" -ForegroundColor Yellow
}

# ===== SUMMARY =====
Write-Host ""
Write-Host "=====================================" -ForegroundColor Cyan
Write-Host "  SUMMARY" -ForegroundColor Cyan
Write-Host "=====================================" -ForegroundColor Cyan
Write-Host ""

if ($totalStopped -gt 0) {
    Write-Host "? Stopped $totalStopped process(es)" -ForegroundColor Green
}

if ($totalFailed -gt 0) {
    Write-Host "? Failed to stop $totalFailed process(es)" -ForegroundColor Red
}

if ($stillRunning.Count -gt 0) {
    Write-Host ""
    Write-Host "? WARNING: Some processes are still running:" -ForegroundColor Yellow
    foreach ($proc in $stillRunning) {
        Write-Host "  � $proc" -ForegroundColor Yellow
    }
    Write-Host ""
    Write-Host "You may need to:" -ForegroundColor Yellow
    Write-Host "1. Close them from Task Manager" -ForegroundColor Gray
    Write-Host "2. Run this script as Administrator" -ForegroundColor Gray
    Log "WARNING: Some processes still running: $($stillRunning -join ', ')" "Yellow"
} else {
    Write-Host "? All services stopped successfully" -ForegroundColor Green
    Log "All services stopped successfully" "Green"
}

if ($totalStopped -eq 0 -and $totalFailed -eq 0) {
    Write-Host "� No services were running" -ForegroundColor Gray
}

Write-Host ""
Write-Host "Press any key to close..." -ForegroundColor Gray
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")