# Auto-fix hosts for ledger.test (requires admin once)
$root = Split-Path -Parent $PSScriptRoot
[int]$port = 3307  # Force this as an Integer
$logsDir = Join-Path $root "logs"
New-Item -ItemType Directory -Force -Path $logsDir | Out-Null
$runLog = Join-Path $logsDir "start_stack.log"

$domain = "ledger.test"
$hostsPath = "$env:WINDIR\System32\drivers\etc\hosts"
$hostsLine = "127.0.0.1`t$domain"

function Is-Admin {
  $id = [Security.Principal.WindowsIdentity]::GetCurrent()
  $p  = New-Object Security.Principal.WindowsPrincipal($id)
  return $p.IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)
}

try {
  [System.Net.Dns]::GetHostAddresses($domain) | Out-Null
} catch {
  if (-not (Is-Admin)) {
    # Relaunch self as admin
    Start-Process powershell -Verb RunAs -ArgumentList "-ExecutionPolicy Bypass -File `"$PSCommandPath`""
    exit
  }

  # Add hosts entry if missing
  $content = Get-Content -Path $hostsPath -ErrorAction Stop
  if ($content -notmatch "(?m)^\s*127\.0\.0\.1\s+$domain\s*$") {
    Add-Content -Path $hostsPath -Value $hostsLine
  }

  ipconfig /flushdns | Out-Null
}

$ErrorActionPreference = "Stop"

function Log($msg) {
  $line = "[{0}] {1}" -f (Get-Date -Format "HH:mm:ss"), $msg
  Write-Host $line
  Add-Content -Path $runLog -Value $line
}

function Require-File($path, $label) {
  if (-not (Test-Path $path)) { throw "$label not found: $path" }
}

try {
  Log "=== START RUN ==="
  Log "Ledger root: $root"

  # Paths
  $caddyExe  = Join-Path $root "bin\caddy.exe"
  $caddyFile = Join-Path $root "conf\Caddyfile"

  $phpDir    = Join-Path $root "php"
  $phpExe    = Join-Path $phpDir "php-cgi.exe"
  $phpIni    = Join-Path $root "conf\php.ini"

  $mariaRoot = Join-Path $root "bin\mariadb"
  $mariaBin  = Join-Path $mariaRoot "bin"
  $mysqld    = Join-Path $mariaBin "mysqld.exe"
  $mariadbd  = Join-Path $mariaBin "mariadbd.exe"
  $installDb = Join-Path $mariaBin "mariadb-install-db.exe"

  $dataDir   = Join-Path $mariaRoot "data"
  $sysDbDir  = Join-Path $dataDir "mysql"
  $dataMyIni = Join-Path $dataDir "my.ini"

  $myTemplate = Join-Path $root "conf\my_template.ini"

  # Checks
  Require-File $caddyExe "Caddy"
  Require-File $phpExe   "PHP"
  Require-File $phpIni   "php.ini"
  Require-File $installDb "mariadb-install-db.exe"
  Require-File $myTemplate "my_template.ini"

  if (Test-Path $mysqld) { $server = $mysqld }
  elseif (Test-Path $mariadbd) { $server = $mariadbd }
  else { throw "No MariaDB server binary found in $mariaBin (need mysqld.exe or mariadbd.exe)" }

  # Ensure Caddyfile exists (safe default)
  if (-not (Test-Path $caddyFile)) {
    New-Item -ItemType Directory -Force -Path (Split-Path $caddyFile) | Out-Null
@"
127.0.0.1:8080

root * "{env.LEDGER_ROOT}\www\ledger"
php_fastcgi 127.0.0.1:9000
file_server
"@ | Set-Content -Path $caddyFile -Encoding ASCII
    Log "Created default Caddyfile"
  }

  # Initialise MariaDB system tables (MariaDB Windows way)
  if (-not (Test-Path $sysDbDir)) {
    Log "Initialising MariaDB data directory..."
    New-Item -ItemType Directory -Force -Path $dataDir | Out-Null

    Push-Location $mariaBin
    try {
      # IMPORTANT (MariaDB 12): use --datadir, --port and --config (template)
      & $installDb --datadir="$dataDir" --port=3307 --config="$myTemplate" --silent 2>&1 |
        Tee-Object -FilePath $runLog -Append
    } finally {
      Pop-Location
    }

    if (-not (Test-Path $sysDbDir)) {
      throw "MariaDB init failed (data\mysql not created)"
    }

    if (-not (Test-Path $dataMyIni)) {
      throw "MariaDB init succeeded but data\my.ini was not created (unexpected)"
    }

    Log "MariaDB initialised OK"
  } else {
    Log "MariaDB already initialised"
  }

  # Start MariaDB using the instance config created in data\my.ini
  Log "Starting MariaDB..."
  Start-Process -FilePath $server `
    -ArgumentList "--defaults-file=`"$dataMyIni`"" `
    -WorkingDirectory $mariaBin `
    -WindowStyle Hidden

# Inside the "Initialise MariaDB system tables" block
if (-not (Test-Path "$sysDbDir")) {
    Log "Initialising MariaDB data directory..."
    # 1. Run the init
    & $installDb --datadir="$dataDir" --silent 2>&1
    
    # 2. Wait 2 seconds for Windows to finish indexing/locking files
    Start-Sleep -Seconds 2
    
    Log "MariaDB initialised OK"
} else {
    Log "MariaDB already exists, skipping init."
}

  Start-Sleep -Seconds 1

  # Start PHP FastCGI
  Log "Starting PHP FastCGI..."
  Start-Process -FilePath $phpExe `
    -ArgumentList "-b 127.0.0.1:9000 -c `"$phpIni`"" `
    -WorkingDirectory $phpDir `
    -WindowStyle Hidden

  Start-Sleep -Milliseconds 500

  # Start Caddy
  $env:LEDGER_ROOT = $root
  Log "Starting Caddy..."
  Start-Process -FilePath $caddyExe `
    -ArgumentList "run --config `"$caddyFile`" --adapter caddyfile" `
    -WorkingDirectory $root `
    -WindowStyle Hidden

  Start-Sleep -Milliseconds 500

# 1. Fix the path for MariaDB (needs forward slashes)
$sqlPath = ("$root\install\schema.sql").Replace('\', '/')

Log "Starting database creation and schema import..."

# 2. Wait for MariaDB to actually be ready (Port check)
Log "Waiting for MariaDB on port $port..."
for ($i=0; $i -lt 10; $i++) {
    if (Test-NetConnection -ComputerName 127.0.0.1 -Port $port -InformationLevel Quiet) { break }
    Start-Sleep -Seconds 1
}

# New variable for the user setup
# 1. Match the name in your SQL dump
$dbName = "ledger_db" 
$createDbQuery = "CREATE DATABASE IF NOT EXISTS $dbName;"

# 2. Update the user query to use the same name
$userQuery = "CREATE USER IF NOT EXISTS 'ledger_user'@'127.0.0.1' IDENTIFIED BY 'ledger123'; GRANT ALL PRIVILEGES ON $dbName.* TO 'ledger_user'@'127.0.0.1'; FLUSH PRIVILEGES;"

try {
    Log "Step 1: Creating Database ($dbName)..."
    & "$root\bin\mariadb\bin\mariadb.exe" -u root -h 127.0.0.1 -P $port --execute="$createDbQuery"

    Log "Step 2: Creating User & Permissions..."
    & "$root\bin\mariadb\bin\mariadb.exe" -u root -h 127.0.0.1 -P $port --execute="$userQuery"

    Log "Step 3: Importing Schema into $dbName..."
    # Notice we use $dbName here instead of ledger_db
    & "$root\bin\mariadb\bin\mariadb.exe" -u root -h 127.0.0.1 -P $port $dbName --execute="SOURCE $sqlPath"

    Log "Database stack fully configured."
}
catch {
    Log "Auto-setup failed: $_"
}

# Log file location
$logFile = "$root\install\import_log.txt"

# Log function
function Log($msg) {
    $timestamp = (Get-Date -Format "yyyy-MM-dd HH:mm:ss")
    $logMessage = "[$timestamp] $msg"
    Add-Content -Path $logFile -Value $logMessage
}

# Log the start of the process
Log "Starting database creation and schema import..."

# Create the database and import the schema, log each step
try {
    # Create the database and log the result
    Log "Executing: $createDbQuery"
    & "$root\bin\mariadb\bin\mysql.exe" -u root -h 127.0.0.1 -P $port -p$password --execute="$createDbQuery" 2>&1 | ForEach-Object { Log $_ }

    # Import the schema and log the result
    Log "Executing: $importSqlQuery"
    & "$root\bin\mariadb\bin\mysql.exe" -u root -h 127.0.0.1 -P $port -p$password ledger_db --execute="$importSqlQuery" 2>&1 | ForEach-Object { Log $_ }

    # Log success
    Log "Database creation and schema import completed successfully."
}
catch {
    # Log error if something fails
    Log "Error: $_"
}

# Define the seed query
# Note: Using 'demo' as the password. MariaDB 10.5+ uses password_hash if provided, 
# or we can insert a pre-hashed string.
# We use single quotes (') so PowerShell leaves the $ signs alone
$liveHash = '$2y$10$QZZ0MEUMSJyMVJsYWkQtJ.uzlXq3Kx0q24CCLjynn5RlbWA1QWb6G'
# 2. Build the query using double quotes, but use the variable for the hash
# PowerShell will insert the hash exactly as it is in the $liveHash variable
$seedQuery = "INSERT INTO users (role, forum_designation, name, email, password_hash) 
              VALUES ('admin', 'Admin', 'James Bond', 'admin@ledger-demo.local', '$liveHash')"

try {
    Log "Step 4: Seeding Admin User..."
    $checkUser = & "$root\bin\mariadb\bin\mariadb.exe" -u root -h 127.0.0.1 -P $port $dbName -N -s -e "SELECT COUNT(*) FROM users WHERE email='admin@ledger-demo.local';"
    
    if ($checkUser -eq 0) {
        # Pass the single-quoted string directly
        & "$root\bin\mariadb\bin\mariadb.exe" -u root -h 127.0.0.1 -P $port $dbName --execute=$seedQuery
        Log "Admin user 'James Bond' created successfully."
    } else {
        Log "Admin user already exists. Skipping seed."
    }
}
catch {
    Log "Seeding failed: $_"
}

# Define the Site Settings Query
# Use @' (single quotes) for literal text
$settingsQuery = @'
INSERT INTO site_settings (id, site_name, site_description, footer_bottom, logo_top, logo_bottom, favicon)
VALUES (1, 'Ledger  Open Support Framework', 'Ledger is an open-source support framework that co...', 'Ledger Support and Knowledge Base', '/img/logo-top.png', '/img/logo-bottom.png', '/img/favicon.png')
ON DUPLICATE KEY UPDATE 
    site_name=VALUES(site_name), 
    site_description=VALUES(site_description);
'@

try {
    Log "Step 5: Seeding Site Settings..."
    # Note: No double-quotes around $settingsQuery here to prevent re-parsing
    & "$root\bin\mariadb\bin\mariadb.exe" -u root -h 127.0.0.1 -P $port $dbName --execute=$settingsQuery
    Log "Site settings applied."
}
catch {
    Log "Settings seed failed: $_"
}

  Log "Stack started"
  Log "Opening https://ledger.test:8443/"
  Start-Process "https://ledger.test:8443/"
}
catch {
  Write-Host ""
  Write-Host "FAILED:"
  Write-Host $_
  Add-Content -Path $runLog -Value ("FAILED: " + ($_ | Out-String))
}
finally {
  Write-Host ""
  Read-Host "Press Enter to close"
}