Set-ExecutionPolicy -Scope Process -ExecutionPolicy Bypass -Force

# ============================================================================
# Ledger Server Diagnostic & Fix Script
# Checks and fixes both MariaDB permissions and PHP configuration issues
# ============================================================================

$root = Split-Path -Parent $PSScriptRoot
$logsDir = Join-Path $root "logs"
New-Item -ItemType Directory -Force -Path $logsDir | Out-Null
$runLog = Join-Path $logsDir "fix_all.log"

function Log($msg, $color = "White") {
  $line = "[{0}] {1}" -f (Get-Date -Format "HH:mm:ss"), $msg
  Write-Host $line -ForegroundColor $color
  Add-Content -Path $runLog -Value $line
}

Write-Host ""
Write-Host "==========================================" -ForegroundColor Cyan
Write-Host "  LEDGER SERVER DIAGNOSTIC & FIX" -ForegroundColor Cyan
Write-Host "==========================================" -ForegroundColor Cyan
Write-Host ""
Log "=== STARTING COMPREHENSIVE DIAGNOSTIC ===" "Cyan"

$issuesFound = @()
$fixesApplied = @()

# ============================================================================
# SECTION 1: STOP ALL SERVICES
# ============================================================================

Write-Host "Step 1: Stopping all services..." -ForegroundColor Cyan
Write-Host ""

$stopped = 0

$caddyProcesses = Get-Process -Name "caddy" -ErrorAction SilentlyContinue
if ($caddyProcesses) {
    foreach ($proc in $caddyProcesses) {
        Stop-Process -Id $proc.Id -Force -ErrorAction SilentlyContinue
        Log "Stopped Caddy (PID: $($proc.Id))" "Gray"
        $stopped++
    }
    Write-Host "  ✓ Stopped Caddy ($($caddyProcesses.Count) process(es))" -ForegroundColor Green
}

$phpProcesses = Get-Process -Name "php-cgi","php" -ErrorAction SilentlyContinue
if ($phpProcesses) {
    foreach ($proc in $phpProcesses) {
        Stop-Process -Id $proc.Id -Force -ErrorAction SilentlyContinue
        Log "Stopped PHP (PID: $($proc.Id))" "Gray"
        $stopped++
    }
    Write-Host "  ✓ Stopped PHP-CGI ($($phpProcesses.Count) process(es))" -ForegroundColor Green
}

$mariaProcesses = Get-Process -Name "mysqld","mariadbd" -ErrorAction SilentlyContinue
if ($mariaProcesses) {
    foreach ($proc in $mariaProcesses) {
        Stop-Process -Id $proc.Id -Force -ErrorAction SilentlyContinue
        Log "Stopped MariaDB (PID: $($proc.Id))" "Gray"
        $stopped++
    }
    Write-Host "  ✓ Stopped MariaDB ($($mariaProcesses.Count) process(es))" -ForegroundColor Green
}

if ($stopped -eq 0) {
    Write-Host "  • No services were running" -ForegroundColor Gray
}

Start-Sleep -Seconds 2

# ============================================================================
# SECTION 2: CHECK MARIADB PERMISSIONS
# ============================================================================

Write-Host ""
Write-Host "Step 2: Checking MariaDB permissions..." -ForegroundColor Cyan
Write-Host ""

$mariaRoot = Join-Path $root "bin\mariadb"
$dataDir = Join-Path $mariaRoot "data"

if (-not (Test-Path $dataDir)) {
    $issuesFound += "MariaDB data directory not found: $dataDir"
    Write-Host "  ✗ Data directory not found" -ForegroundColor Red
} else {
    Log "Checking MariaDB data directory: $dataDir" "Gray"
    
    # Critical files to check
    $criticalFiles = @(
        "ibdata1",
        "ibtmp1",
        "ib_buffer_pool",
        "undo001",
        "undo002",
        "undo003",
        "aria_log_control",
        "aria_log.00000001",
        "tc.log"
    )
    
    $readOnlyCount = 0
    $fixedCount = 0
    
    foreach ($fileName in $criticalFiles) {
        $filePath = Join-Path $dataDir $fileName
        
        if (Test-Path $filePath) {
            $file = Get-Item $filePath
            
            if ($file.IsReadOnly) {
                $readOnlyCount++
                Write-Host "  ⚠ $fileName is READ-ONLY" -ForegroundColor Yellow
                
                try {
                    $file.IsReadOnly = $false
                    Write-Host "    ✓ Fixed" -ForegroundColor Green
                    $fixedCount++
                    $fixesApplied += "Removed read-only from $fileName"
                } catch {
                    Write-Host "    ✗ Failed to fix" -ForegroundColor Red
                    $issuesFound += "Could not remove read-only from $fileName"
                }
            }
        }
    }
    
    # Check all files recursively
    $allFiles = Get-ChildItem -Path $dataDir -Recurse -File -ErrorAction SilentlyContinue
    $totalReadOnly = 0
    
    foreach ($file in $allFiles) {
        if ($file.IsReadOnly) {
            $totalReadOnly++
            try {
                $file.IsReadOnly = $false
                $fixedCount++
            } catch {
                # Silently continue for files we can't fix
            }
        }
    }
    
    if ($totalReadOnly -gt 0) {
        Write-Host "  ⚠ Found $totalReadOnly read-only file(s) in data directory" -ForegroundColor Yellow
        Write-Host "    ✓ Fixed $fixedCount file(s)" -ForegroundColor Green
        $fixesApplied += "Fixed $fixedCount read-only files in MariaDB data directory"
    } else {
        Write-Host "  ✓ All MariaDB files are writable" -ForegroundColor Green
    }
    
    # Set directory permissions
    try {
        $acl = Get-Acl $dataDir
        $currentUser = [System.Security.Principal.WindowsIdentity]::GetCurrent().Name
        $permission = $currentUser, "FullControl", "ContainerInherit,ObjectInherit", "None", "Allow"
        $accessRule = New-Object System.Security.AccessControl.FileSystemAccessRule $permission
        $acl.SetAccessRule($accessRule)
        Set-Acl $dataDir $acl
        Write-Host "  ✓ Set full control permissions on data directory" -ForegroundColor Green
        $fixesApplied += "Set full control permissions on MariaDB data directory"
    } catch {
        $issuesFound += "Could not set directory permissions (may need Administrator)"
    }
}

# ============================================================================
# SECTION 3: CHECK PHP CONFIGURATION
# ============================================================================

Write-Host ""
Write-Host "Step 3: Checking PHP configuration..." -ForegroundColor Cyan
Write-Host ""

$phpDir = Join-Path $root "php"
$phpExe = Join-Path $phpDir "php-cgi.exe"
$phpIni = Join-Path $root "conf\php.ini"
$extDir = Join-Path $phpDir "ext"

# Check if PHP exists
if (-not (Test-Path $phpExe)) {
    $issuesFound += "PHP-CGI not found: $phpExe"
    Write-Host "  ✗ PHP-CGI executable not found" -ForegroundColor Red
} else {
    Write-Host "  ✓ PHP-CGI found" -ForegroundColor Green
}

# Check php.ini
if (-not (Test-Path $phpIni)) {
    $issuesFound += "php.ini not found: $phpIni"
    Write-Host "  ✗ php.ini not found" -ForegroundColor Red
    
    # Try to create from template
    $phpIniDev = Join-Path $root "conf\php.ini-development"
    $phpIniProd = Join-Path $root "conf\php.ini-production"
    
    if (Test-Path $phpIniDev) {
        try {
            Copy-Item $phpIniDev $phpIni
            Write-Host "    ✓ Created php.ini from php.ini-development" -ForegroundColor Green
            $fixesApplied += "Created php.ini from template"
        } catch {
            $issuesFound += "Could not create php.ini from template"
        }
    } elseif (Test-Path $phpIniProd) {
        try {
            Copy-Item $phpIniProd $phpIni
            Write-Host "    ✓ Created php.ini from php.ini-production" -ForegroundColor Green
            $fixesApplied += "Created php.ini from template"
        } catch {
            $issuesFound += "Could not create php.ini from template"
        }
    }
}

# Check PHP extensions
if (Test-Path $phpIni) {
    Write-Host "  ✓ php.ini found" -ForegroundColor Green
    
    $content = Get-Content $phpIni -Raw
    $requiredExtensions = @("mysqli", "mbstring", "openssl", "pdo_mysql", "curl")
    
    Write-Host ""
    Write-Host "  Checking required extensions:" -ForegroundColor Gray
    
    $needsUpdate = $false
    $extensionsToFix = @()
    
    foreach ($ext in $requiredExtensions) {
        if ($content -match "^extension=$ext" -or $content -match "^extension=php_$ext\.dll") {
            Write-Host "    ✓ $ext enabled" -ForegroundColor Green
        } elseif ($content -match ";extension=$ext" -or $content -match ";extension=php_$ext\.dll") {
            Write-Host "    ⚠ $ext disabled (commented out)" -ForegroundColor Yellow
            $needsUpdate = $true
            $extensionsToFix += $ext
        } else {
            Write-Host "    ✗ $ext not found in php.ini" -ForegroundColor Red
            $needsUpdate = $true
            $extensionsToFix += $ext
        }
    }
    
    if ($needsUpdate) {
        Write-Host ""
        Write-Host "  Fixing PHP extensions..." -ForegroundColor Yellow
        
        # Backup php.ini
        $backupPath = "$phpIni.backup_$(Get-Date -Format 'yyyyMMdd_HHmmss')"
        Copy-Item $phpIni $backupPath
        Log "Created backup: $backupPath" "Gray"
        
        # Read as array of lines
        $lines = Get-Content $phpIni
        $newLines = @()
        
        foreach ($line in $lines) {
            $modified = $false
            
            # Uncomment disabled extensions
            foreach ($ext in $extensionsToFix) {
                if ($line -match "^;extension=$ext" -or $line -match "^;extension=php_$ext\.dll") {
                    $newLines += $line -replace "^;", ""
                    Write-Host "    ✓ Enabled $ext" -ForegroundColor Green
                    $fixesApplied += "Enabled PHP extension: $ext"
                    $modified = $true
                    break
                }
            }
            
            if (-not $modified) {
                $newLines += $line
            }
        }
        
        # Add missing extensions
        foreach ($ext in $extensionsToFix) {
            $found = $false
            foreach ($line in $newLines) {
                if ($line -match "^extension=$ext" -or $line -match "^extension=php_$ext\.dll") {
                    $found = $true
                    break
                }
            }
            
            if (-not $found) {
                $newLines += "extension=$ext"
                Write-Host "    ✓ Added $ext" -ForegroundColor Green
                $fixesApplied += "Added PHP extension: $ext"
            }
        }
        
        # Write back
        $newLines | Set-Content $phpIni
        Log "Updated php.ini with required extensions" "Green"
    }
}

# Check extension DLL files
Write-Host ""
Write-Host "  Checking extension DLL files:" -ForegroundColor Gray

if (-not (Test-Path $extDir)) {
    $issuesFound += "PHP ext directory not found: $extDir"
    Write-Host "    ✗ Extension directory not found" -ForegroundColor Red
} else {
    $requiredExtensions = @("mysqli", "mbstring", "openssl", "pdo_mysql", "curl")
    
    foreach ($ext in $requiredExtensions) {
        $dllPath = Join-Path $extDir "php_$ext.dll"
        if (Test-Path $dllPath) {
            Write-Host "    ✓ php_$ext.dll exists" -ForegroundColor Green
        } else {
            Write-Host "    ✗ php_$ext.dll NOT FOUND" -ForegroundColor Red
            $issuesFound += "Missing PHP extension DLL: php_$ext.dll"
        }
    }
}

# ============================================================================
# SECTION 4: CHECK PORT AVAILABILITY
# ============================================================================

Write-Host ""
Write-Host "Step 4: Checking port availability..." -ForegroundColor Cyan
Write-Host ""

$portsToCheck = @(
    @{Port=3307; Service="MariaDB"},
    @{Port=9000; Service="PHP-CGI"},
    @{Port=8443; Service="Caddy HTTPS"},
    @{Port=8080; Service="Caddy HTTP"}
)

foreach ($portInfo in $portsToCheck) {
    try {
        $connection = New-Object System.Net.Sockets.TcpClient
        $connection.Connect("127.0.0.1", $portInfo.Port)
        $connection.Close()
        Write-Host "  ⚠ Port $($portInfo.Port) ($($portInfo.Service)) is in use" -ForegroundColor Yellow
        $issuesFound += "Port $($portInfo.Port) is already in use by another process"
    } catch {
        Write-Host "  ✓ Port $($portInfo.Port) ($($portInfo.Service)) is available" -ForegroundColor Green
    }
}

# ============================================================================
# SUMMARY
# ============================================================================

Write-Host ""
Write-Host "==========================================" -ForegroundColor Cyan
Write-Host "  DIAGNOSTIC SUMMARY" -ForegroundColor Cyan
Write-Host "==========================================" -ForegroundColor Cyan
Write-Host ""

if ($fixesApplied.Count -gt 0) {
    Write-Host "FIXES APPLIED:" -ForegroundColor Green
    foreach ($fix in $fixesApplied) {
        Write-Host "  ✓ $fix" -ForegroundColor Green
        Log "FIX: $fix" "Green"
    }
    Write-Host ""
}

if ($issuesFound.Count -gt 0) {
    Write-Host "REMAINING ISSUES:" -ForegroundColor Yellow
    foreach ($issue in $issuesFound) {
        Write-Host "  ⚠ $issue" -ForegroundColor Yellow
        Log "ISSUE: $issue" "Yellow"
    }
    Write-Host ""
    
    Write-Host "RECOMMENDATIONS:" -ForegroundColor Cyan
    
    if ($issuesFound -match "Missing PHP extension DLL") {
        Write-Host "  • PHP installation is incomplete - download full PHP package" -ForegroundColor Gray
        Write-Host "    from https://windows.php.net/download/" -ForegroundColor Gray
    }
    
    if ($issuesFound -match "Port.*already in use") {
        Write-Host "  • Check Task Manager for processes using these ports" -ForegroundColor Gray
        Write-Host "  • Another instance of services may be running" -ForegroundColor Gray
    }
    
    if ($issuesFound -match "Administrator") {
        Write-Host "  • Run this script as Administrator for full permissions" -ForegroundColor Gray
    }
    
} else {
    Write-Host "✓ NO ISSUES FOUND - All checks passed!" -ForegroundColor Green
    Log "All diagnostic checks passed" "Green"
}

Write-Host ""
Write-Host "NEXT STEPS:" -ForegroundColor Cyan
if ($fixesApplied.Count -gt 0 -or $stopped -gt 0) {
    Write-Host "  1. Changes were made - services need to be restarted" -ForegroundColor Gray
    Write-Host "  2. Run start_server.ps1 to start all services" -ForegroundColor Gray
    Write-Host "  3. Test your application" -ForegroundColor Gray
} else {
    Write-Host "  1. Run start_server.ps1 to start all services" -ForegroundColor Gray
    Write-Host "  2. If problems persist, review issues above" -ForegroundColor Gray
}

Write-Host ""
Write-Host "Log file: $runLog" -ForegroundColor Gray
Write-Host ""
Write-Host "Press any key to close..." -ForegroundColor Gray
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")